/* Adobe.Walk.js -- Walk behavior for Animal */

define(["lib/Zoot", 
	"lodash", "lib/dev", "lib/tasks", "src/animate/faceUtils"],
function (Z, 
	lodash, dev, tasks, faceUtils) {
	"use strict";
	
	var mat3 = Z.Mat3, vec2 = Z.Vec2,
		kGaitStyle_HeadBang = 0,
		aGaitStyles = [
			// first is default for old projects if an unknown id is used
			{ id: 3,	uiName: "$$$/animal/Behavior/Walk/Style/Walk=Walk", walkFile: "default_walk" },
			{ id: 10,	uiName: "$$$/animal/Behavior/Walk/Style/Slump=Slump", walkFile: "walk_down" },
			{ id: 7,	uiName: "$$$/animal/Behavior/Walk/Style/Strut=Strut", walkFile: "squat" },
			{ id: 11,	uiName: "$$$/animal/Behavior/Walk/Style/Prance=Prance", walkFile: "walk_high" },
			{ id: 6,	uiName: "$$$/animal/Behavior/Walk/Style/Sneak=Sneak", walkFile: "sneak" },
			{ id: 5,	uiName: "$$$/animal/Behavior/Walk/Style/Run=Run", walkFile: "run" },
			{ id: kGaitStyle_HeadBang,	uiName: "$$$/animal/Behavior/Walk/Style/HeadBang=Head Bang", walkFile: "bend_waist" },
		],
		
		kLegFieldNames = [
				"LeftAnkle",
				"LeftToe",
				"LeftHeel",
				"LeftKnee",
				"RightAnkle",
				"RightToe",
				"RightHeel",
				"RightKnee",
		],
		
		kMoveEntirePuppet = 0,
		kMoveHandles = 1,
		
		kStartImmediately = 0,
		kStartWithArrowKeys = 1,
		
		kFacingAutomatic = 0,
		kFacingLeft = 1,
		kFacingRight = 2,

		aHandleTagDefinitions = [
			{
				id: "Adobe.Walk.Hip",
				artMatches: ["hip"],
				uiName: "$$$/animal/Behavior/Walk/TagName/Hip=Hip",
			},
			{
				id: "Adobe.Walk.RightKnee",
				artMatches: ["right knee"],
				uiName: "$$$/animal/Behavior/Walk/TagName/RightKnee=Right Knee",
			},
			{
				id: "Adobe.Walk.RightAnkle",
				artMatches: ["right ankle"],
				uiName: "$$$/animal/Behavior/Walk/TagName/RightAnkle=Right Ankle",
			},
			{
				id: "Adobe.Walk.RightHeel",
				artMatches: ["right heel"],
				uiName: "$$$/animal/Behavior/Walk/TagName/RightHeel=Right Heel",
			},
			{
				id: "Adobe.Walk.RightToe",
				artMatches: ["right toe"],
				uiName: "$$$/animal/Behavior/Walk/TagName/RightToe=Right Toe",
			},
			{
				id: "Adobe.Walk.LeftKnee",
				artMatches: ["left knee"],
				uiName: "$$$/animal/Behavior/Walk/TagName/LeftKnee=Left Knee",
			},
			{
				id: "Adobe.Walk.LeftAnkle",
				artMatches: ["left ankle"],
				uiName: "$$$/animal/Behavior/Walk/TagName/LeftAnkle=Left Ankle",
			},
			{
				id: "Adobe.Walk.LeftHeel",
				artMatches: ["left heel"],
				uiName: "$$$/animal/Behavior/Walk/TagName/LeftHeel=Left Heel",
			},
			{
				id: "Adobe.Walk.LeftToe",
				artMatches: ["left toe"],
				uiName: "$$$/animal/Behavior/Walk/TagName/LeftToe=Left Toe",
			},
			{
				id: "Adobe.Walk.Waist",
				artMatches: ["waist"],
				uiName: "$$$/animal/Behavior/Walk/TagName/Waist=Waist",
			},
			{
				id: "Adobe.Walk.Neck",
				artMatches: ["neck"],
				uiName: "$$$/animal/Behavior/Walk/TagName/Neck=Neck",
			},
			{
				id: "Adobe.Walk.RightElbow",
				artMatches: ["right elbow"],
				uiName: "$$$/animal/Behavior/Walk/TagName/RightElbow=Right Elbow",
			},
			{
				id: "Adobe.Walk.RightWrist",
				artMatches: ["right wrist"],
				uiName: "$$$/animal/Behavior/Walk/TagName/RightWrist=Right Wrist",
			},
			{
				id: "Adobe.Walk.LeftElbow",
				artMatches: ["left elbow"],
				uiName: "$$$/animal/Behavior/Walk/TagName/LeftElbow=Left Elbow",
			},
			{
				id: "Adobe.Walk.LeftWrist",
				artMatches: ["left wrist"],
				uiName: "$$$/animal/Behavior/Walk/TagName/LeftWrist=Left Wrist",
			},
			
			// no longer defining Noggin tag; plan was to switch to existing
			//	Head tag, but it fights with the Face behavior and doesn't
			//	add much control over having Neck, so I'm leaving it out for now.
			//	Note that "Noggin" in LocomotionComponent.cpp was updated to "Head"
			//	but we're no longer reading or writing it from this behavior.
		];
	
	aHandleTagDefinitions.forEach (function (tag) {
		tag.tagType = "handletag";
		tag.uiGroups = [{ id:"Adobe.TagGroup.Body"}];			
	});
	
	var aViewTags = [];
	
	faceUtils.viewLayerTagDefinitions.forEach(function (tag) {
		// these tags REPEATED below; TODO: factor
		if (tag.id === "Adobe.Face.LeftProfile" || tag.id === "Adobe.Face.RightProfile") {
			aViewTags.push(tag);
		}
	});
		
	var aAllTagDefinitions = aHandleTagDefinitions.concat(aViewTags);
	
	function deepClone (obj) {
		return JSON.parse(JSON.stringify(obj));
	}
		
	// $$$ TODO: factor with FaceTracker:
	function makeValidIdFromLabel (str) {
		return str.replace(/[^\w]/g, "_");
	}
	
	function makeHandleIdFromLabel (str) {
		return "H_" + makeValidIdFromLabel(str);
	}

	
	function defineHandleParams () {
		var aParams = [];
		
		aHandleTagDefinitions.forEach( function (tag) {
			var label = tag.id;
			var def = {id:makeHandleIdFromLabel(label), type:"handle", uiName:tag.uiName,
						dephault:{match:"//"+label},
						maxCount:1,
						hidden:false};
			
			def.maxCount = 1;
			def.dephault.startMatchingAtParam = "viewLayers";
			
			aParams.push(def);
		});
		
		return aParams;
	}
	
	
	function getShortIDFromTagID (tagID) {
		return tagID.split(".")[2];
	}
	
	
	function invertOrIndentity (m, sLogFailureMsg0) {
		try {
			return Z.Mat3.invert(m);
		} catch (e) {
			if (sLogFailureMsg0) {
				console.logToUser(sLogFailureMsg0);
			}
			return Z.Mat3.identity();
		}
	}
	
	function toePointingLeft (toe, heel, ankle) {
		var heelOrAnkle = heel || ankle;	// prefer heel, but ankle OK too
		if (toe && heelOrAnkle) {			// needs both
			return toe[0] < heelOrAnkle[0];	// check x-coord
		} else {
			return undefined;
		}
	}
	
	
	function flipHorizontal (positions) {
		Object.keys(positions).forEach(function (key) {
			positions[key][0] *= -1;	// flip x-coord to make pose face right
		});
	}
	
	// clockwise angle in radians; needs discontinuity at 12 o'clock (dx=0, dy=-1), so
	//	zero is 6pm
	function angleFromPoints (p1, p2) {
		// unusual swapping of arguments and negating of result to meet the 
		//	clockwise/north-discontinuity requirements
		return -Math.atan2(p2[0] - p1[0], p2[1] - p1[1]);
	}
	
//	function testAngle(a, b)
//	{
//		console.logToUser(JSON.stringify(b) + " = " + angleFromPoints(a, b));
//	}
//	
//		testAngle([0,0], [1,0]);
//		testAngle([0,0], [0,1]);
//		testAngle([0,0], [-1,0]);
//		testAngle([0,0], [-1,-1]);
//		testAngle([0,0], [1,1]);
//		testAngle([0,0], [0,-1]);
//
	
	function kneeBentToLeft (knee, heel, ankle, hip, waist, neck) {
		var bLeft,	// undefined
			body = hip || waist || neck,
			foot = heel || ankle;
		
		if (body && knee && foot) {
			var kneeAngle = angleFromPoints(body, knee),	// clockwise angle
				footAngle = angleFromPoints(body, foot);
			
			bLeft = (kneeAngle > footAngle);
		}
		
		return bLeft;
	}
	
	
	// returns true if facing left, undefined if not sure
	function checkFacingDirection (ps /* positions object */, facingParam)
	{
		var bFacingLeft;
				
		switch (facingParam) {
			default:
			case kFacingAutomatic:
				// first try feet
				bFacingLeft = toePointingLeft(ps.RightToe, ps.RightHeel, ps.RightAnkle);
				
				if (bFacingLeft === undefined) {
					bFacingLeft = toePointingLeft(ps.LeftToe, ps.LeftHeel, ps.LeftAnkle);
				}
				
				// if that doesn't work, try knee bending direction
				if (bFacingLeft === undefined) {
					bFacingLeft = kneeBentToLeft(ps.RightKnee, ps.RightHeel, ps.RightAnkle,
												ps.RightHip, ps.Waist, ps.Neck);
				}
				
				if (bFacingLeft === undefined) {
					bFacingLeft = kneeBentToLeft(ps.LeftKnee, ps.LeftHeel, ps.LeftAnkle,
												ps.LeftHip, ps.Waist, ps.Neck);
				}
				
				/* now allowing undefined to be returned to indicate unknown, but not
					using that info yet (still falsy, no net change)
					if (bFacingLeft === undefined) {
						bFacingLeft = false;
					}
				*/
				break;
				
			case kFacingLeft:
				bFacingLeft = true;
				break;
			
			case kFacingRight:
				bFacingLeft = false;
				break;
		}
		
		//console.logToUser("checkFacingDirection = " + bFacingLeft);
		
		return bFacingLeft;
	}
    
    
    function handleIdToLoco (id)        // for _reading_ from loco fields (for writing, you have to write to both left & right)
    {
        if (id === "Hip") {
            return "RightHip";
        }
        return id;
    }
	
	
	// locomotion code gets particularly annoyed if no right ankle is specified
	function fabricateAnkleFromHeelIfNeeded (	sSide, 		// "Left" or "Right"
												positions) 	// inout
	{
		var sAnkle = sSide + "Ankle",
			kFactor = 0.15,	// proportional distance of heel->ankle relative to root
			ankle = positions[sAnkle],
			heel = positions[sSide + "Heel"] || positions[sSide + "Toe"];	// use toe if heel not available
		
		if (!ankle && heel) {
			// can always depend on having Root
			var heelToRoot = vec2.subtract(positions.Root, heel);
			
			positions[sAnkle] = vec2.add(heel, vec2.scale(kFactor, heelToRoot));
		}
	}

	
	// pass two points
	function distanceBetween (a, b) {
		return vec2.magnitude(vec2.subtract(a, b));
	}
	
	
	// returns number or undefined
	function getForearmLen (wrist, elbow) {
		if (wrist && elbow) {
			return distanceBetween(wrist, elbow);
		}
	}


	// arms can't swing without a neck; to fabricate it needs a wrist/elbow pair and a hip or waist
	function fabricateNeckFromHipIfNeeded (positions)	// inout
	{
		if (!positions.Neck) {
			var hip = positions.RightHip || positions.Waist,
				forearmLen;
			
			if (hip) {
				forearmLen =	getForearmLen(positions.RightWrist,	positions.RightElbow) ||
								getForearmLen(positions.LeftWrist,	positions.LeftElbow);
				
				if (forearmLen) {
					// approximating neck position by assuming it's twice the forearm distance
					//	above the hip
					positions.Neck = vec2.add(hip, [0, -forearmLen*2]);
				}
			}
		}
	}
	
	
	function getRootPosition (positions)
	{
		if (positions.RightHip && positions.Waist) {
			return Z.Vec2.scale(0.5, Z.Vec2.add(positions.RightHip, positions.Waist)); // midpoint
		} else {
			var root = positions.RightHip || positions.Waist;
			
			if (root) {
				return root;
			} else if (positions.Neck) {
				return [positions.Neck[0], 0];
			}
		}
		
		// last resort
		return [0, 0];	// or perhaps we should do the center of the
						//	bounding box of all position handles?
	}
	
	
	// returns { floorY (number), sFloorHandleName (string), positions (object mapping bodypart -> array[2]) }
	// floorY is the max Y-coord of all defined handles; sFloorHandleName indicates that handle
	function gatherInitialHandlePositions (args, aHandleIDs, handles) {
		//	we want something like getHandleFrames() but for all the handles in our handle params
		//	and all need to be in the same coordinate frame
		var maxY,
			sMaxHandleName,
			positions = {},
            locations = {}, // for screening out duplicates
			stageLayer = args.stageLayer.privateLayer,
			handlePuppet = stageLayer.getHandleTreeRoot(),
			matStageLayer_Scene = invertOrIndentity(mat3.multiply(args.getLayerMatrixRelativeToScene(stageLayer),
																  tasks.handle.getFrameRelativeToLayer(handlePuppet)),
				"Walk behavior working around initial zero-scale -- some gait calculations may be wacky.");
		
		/* in practice, invertOrIndentity() will fail back to identity when the puppet's Transform scale is zero;
			this means we won't have a common frame of reference for the handles, so initial measurements
			will be wrong if the handles aren't all on the same warper layer. It is possible to get all the
			handles into a common frame without going through scene coordinate, which means avoiding the
			singular scale. IOW, we could calculate matStageLayer_Layer without going through Scene.
			So if it becomes a problem that zero-scaled puppets (at the start time) don't Walk correctly,
			we'll need to make that calculation change here.
		*/

		aHandleIDs.forEach(function (id) {
			// for example, set positions["LeftKnee"] = handles["Adobe.Walk.LeftKnee"]
			var shortID = getShortIDFromTagID(id);

			var h = handles[id],
				matStageLayer_Layer = Z.Mat3.multiply(matStageLayer_Scene, args.getLayerMatrixRelativeToScene(h.getWarperLayer())),
				mat = Z.Mat3.multiply(matStageLayer_Layer, tasks.handle.getFrameRelativeToLayer(h)),
				pos = mat.getTranslation(),	// Array[2]
                sPos = JSON.stringify(pos);
            
            if (sPos in locations) {
                console.logToUser("two body handles (" + shortID + " & " + locations[sPos] + ") are at the same location, " + shortID + " will be ignored.");
            } else {
                locations[sPos] = shortID;
            
                if (maxY === undefined || maxY < pos[1]) {
                    maxY = pos[1];
                    sMaxHandleName = handleIdToLoco(shortID);
                }
                //console.logToUser("construction: " + matScene_Layer);

                //pos = Z.Vec2.transformAffine(mat, [0,0]);

                if (shortID === "Hip") {
                    // Locomotion code sends/receives identical LeftHip & RightHip, but
                    //	for our handles we only have "Hip"
                    positions.LeftHip = pos;
                    positions.RightHip = pos;
                } else {
                    positions[shortID] = pos;
                }
            }
		});
		
		positions.Root = getRootPosition(positions);

		fabricateAnkleFromHeelIfNeeded("Left", positions);
		fabricateAnkleFromHeelIfNeeded("Right", positions);
		
		fabricateNeckFromHipIfNeeded(positions);
		
		// not clear why, but we have to shift all coordinates so that root is at zero
		//	otherwise when the top-level puppet origin doesn't line up with the center of
		//	the body handles everything gets offset (DVACH-$$$$)
		var offset = deepClone(positions.Root);
		for (var tag in positions) {
			if (positions.hasOwnProperty(tag)) {
				// important that this makes a copy, as the position array info
				//	sometimes shares arrays
				positions[tag] = Z.Vec2.subtract(positions[tag], offset);
			}
		}
		maxY -= offset[1];	// apply same offset to maxY so it's in the same coord system
				
		return { floorY: maxY, sFloorHandleName: sMaxHandleName, positions: positions, rootOffset: offset };
	}
	
	
	// returns phaseAdjustment (nonzero only for fresh gait setups)
	function setupGait (self, args, viewData)
	{
		var phaseAdjustment = 0,
			gait = args.getParam("gaitStyle"),
			bCushion = args.getParam("cushion"),
			facingParam = args.getParam("facing"),
			bSetOverallMatrix = (args.getParam("overallMovement") === kMoveEntirePuppet);

		if (viewData.walkState === undefined || viewData.gait !== gait || viewData.bCushion !== bCushion ||
			viewData.facingParam !== facingParam ||
			viewData.bSetOverallMatrix !== bSetOverallMatrix) {

			if (viewData.bSetOverallMatrix !== bSetOverallMatrix) {
				// this makes switching from kMoveEntirePuppet to kMoveHandles always give
				//	the same result; but switching the other way can still have other handles
				//	end up far from home. Temporary anyway.
				tasks.handle.setFrame(args.stageLayer.privateLayer.getHandleTreeRoot(),
				Z.Mat3.identity(),
				tasks.dofs.type.kAffine);
			}

			viewData.gait = gait;
			viewData.bCushion = bCushion;
			viewData.facingParam = facingParam;
			viewData.bSetOverallMatrix = bSetOverallMatrix;

			var gaitFileName;

			aGaitStyles.forEach(function (gs) {
				if (gs.id === gait) {
					gaitFileName = gs.walkFile;
				}
			});

			if (gaitFileName === undefined) {	// loaded from older project with different options?
				gaitFileName = aGaitStyles[0].walkFile;	// default to first
			}
			
			var positions = viewData.initialPositions;	// initialPositions is always original (unflipped coords)
			
			viewData.bFacingLeft = checkFacingDirection(positions, facingParam);

			if (viewData.bFacingLeft) {
				positions = deepClone(positions);
				flipHorizontal(positions);
				phaseAdjustment = 0.5;	// so switching between left & right views looks better
			}

			viewData.walkState = nmlImpl.newLoco(gaitFileName + ".txt", bCushion, positions);
			viewData.lastRootX = undefined;
			
			// now run through one cycle to figure out the lowest point reached by the
			//	floor handle, so we know the proper offset to have it match the rest pose
			//	(normally this is a toe or heel handle, but they might not exist)
			
			var numFramesToSampleInCycle = 16,			// empirically this is enough to be within a pixel of truth for our 7 gaits
				deltaT = 1.0/numFramesToSampleInCycle,	// sample across one second to get full cycle
				maxY;
			
			for (var step = 0; step < numFramesToSampleInCycle; ++step) {
				var newValues = viewData.walkState.step(deltaT),
                    pos = newValues[viewData.sFloorHandleName];
                
                if (pos) {
					var y = pos[1];

                    if (maxY === undefined || y > maxY) {
                        maxY = y;
                    }
                } else {
                    // locomotion code got a NaN and bailed; perhaps we should do a loud user error?
                }
			}
            
            viewData.walkState.step(-1.0);  // put simulation back to time zero, though probably doesn't matter
						
			viewData.floorHandleY = maxY;
		}
		
		return phaseAdjustment;
	}
    
    
	function attachTasksToHandles(self, args, viewData, matScene_StageLayer, aHandleIDs, newValues, toeCurl, strength)
	{
		aHandleIDs.forEach(function (id) {
			var shortID = getShortIDFromTagID(id);

            shortID = handleIdToLoco(shortID);
			
			var pos = newValues[shortID];

			if (pos) { // in sceneLayer space, convert to handle-local

				var h = viewData.handles[id],
					matScene_Layer = args.getLayerMatrixRelativeToScene(h.getWarperLayer()),
					matLayer_Scene = invertOrIndentity(matScene_Layer),	// zero-scale here likely to be invisible anyway, so no worries if we get back identity
					matLayer_StageLayer = Z.Mat3.multiply(matLayer_Scene, matScene_StageLayer),
					posInLayer = Z.Vec2.transformAffine(matLayer_StageLayer, pos),
					handlePos;
				
				try {
					handlePos = tasks.handle.convertLayerPoint(h, posInLayer);

					//console.logToUser("update: " + matScene_Layer);
					
					if (false) {
						// non-Task method (for debugging)
						tasks.handle.setFrame(h, Z.Mat3.translation(handlePos), 2 /* use translation only */);
					} else {
						var transform = {
							x: handlePos[0],
							y: handlePos[1]
						};
						
						var moveTo = new tasks.MoveTo(transform);

						tasks.handle.attachTask(h, moveTo, strength);
						
						if (shortID === "LeftToe" || shortID === "RightToe") {
							var field = (shortID === "LeftToe" ? "LeftToeAngle" : "RightToeAngle"),
								toeAngle = toeCurl[field];
							
							if (toeAngle !== undefined) {
								var rotateTo = new tasks.MoveTo({angle:toeAngle});

								tasks.handle.attachTask(h, rotateTo, strength);	
							}
						}
						

					}
				} catch (e) {
					// convertLayerPoint probably failed due to zero scale; just skip this handle
					console.logToUser("Walk skipping " + shortID + " handle: " + e);
				}
			}
		});
	}
	
	// returns { direction: (-1, 0, 1), stepSpeed: (number), bCheckArrows: (boolean) }
	function getUserCommandedDirection (self, args, t)	// independent of view
	{
		var bCheckArrows = (args.getParam("start") === kStartWithArrowKeys),
			triggerJustChanged = "",	// or START/END (i.e. just pressed or released)
			bMovingOK = true,
			direction,
			strength = 1,
			stepSpeed = args.getParam("speed")/100;

		if (bCheckArrows) {
			var kc = Z.keyCodes,
				rightDown = args.getParamEventValue("keyInput", kc.getKeyGraphId(kc.rightKey)),
				leftDown = 	args.getParamEventValue("keyInput", kc.getKeyGraphId(kc.leftKey));

			if (rightDown || leftDown) {
				args.setEventGraphParamRecordingValid("keyInput");

				direction = leftDown ? -1 : 1;
				self.oldDirection = direction;
								
				if (!self.bKeyPreviouslyDown) {
					triggerJustChanged = "START";
					self.bKeyPreviouslyDown = true;
				}
			} else {
				if (self.bKeyPreviouslyDown) {
					triggerJustChanged = "END";
					self.bKeyPreviouslyDown = false;
				}
				direction = self.oldDirection;
				bMovingOK = false;		// later conditionalized on easing
			}
		} else {
			direction = 1;	// for Immediately, default is right (later flipped if drawn left)
		}
		
		var easeDur = args.getParam("ease");

		if (triggerJustChanged && easeDur > 0) {
			if (self.triggerPhase) {
				// previous trigger didn't finish, so shorten the ease time
				//	accordingly by skipping the first portion (keeps it continuous)
				//	IOW, start the triggerTime before now
				self.triggerTime = t - easeDur + t - self.triggerTime;
			} else {
				self.triggerTime = t;
			}
			self.triggerPhase = triggerJustChanged;
		}
		
		if (self.triggerPhase) {
			var progress = (t - self.triggerTime) / easeDur;
						
			if (progress < 1) {
				if (progress < 0) {
					progress = 0;
				}
				if (self.triggerPhase === "END") {
					progress = 1 - progress;
					bMovingOK = true;
				}
				stepSpeed *= progress;
			} else {
				self.triggerTime = undefined;
				self.triggerPhase = undefined;
			}
			strength = progress;
		}

		if (!bMovingOK) {
			stepSpeed = 0;
			direction = 0;
			strength = 0;
		}

		return { direction: direction, stepSpeed: stepSpeed, strength: strength, bCheckArrows: bCheckArrows };
	}
	
	
	function measureArm (neck, elbow, wrist) {
		var m;	// can return undefined
		
		if (neck) {
			elbow = elbow || wrist;	// if no elbow, treat wrist like an elbow
			
			if (elbow) {
				m = {
					upperLen: distanceBetween(neck, elbow)
				};
				
				if (wrist) {
					m.forearmLen = distanceBetween(elbow, wrist);
				}
			}
		}
		
		return m;
	}

	
	function calcElbowAndWrist (armMeasurements, armAngle, elbowBend, armSwing,
								knee, ankle, heel, toe,
								v,	// read-write (wrist elbow & wrist field names)
								elbowFieldName, wristFieldName)
	{
		var radians = Z.mathUtils.rad,
			hip = v.Hip || v.Waist || v.Neck,
			foot = ankle || heel || toe || knee,
			centerOfRotation = v.Neck;
		
		if (armMeasurements && hip && foot && centerOfRotation) {
			var angle = angleFromPoints(hip, foot)*armSwing,
				upperXform = mat3.rotation(angle + radians(armAngle)).pretranslate(v.Neck),	// from Neck-origin -> puppet
				forearmXform = mat3.rotation(angle + radians(-elbowBend)).pretranslate([0, armMeasurements.upperLen]); // from Elbow-origin -> Neck-origin

			v[elbowFieldName] = vec2.transformAffine(upperXform, [0, armMeasurements.upperLen]);
			if (armMeasurements.forearmLen !== undefined) {
				v[wristFieldName] = vec2.transformAffine(upperXform,
												 vec2.transformAffine(forearmXform, [0, armMeasurements.forearmLen]));
			}
		}
	}
	
	
	// eventually the locomotion code should produce this data via its internal keyframes;
	//	for now, elbows & wrists are based on angle from hip to foot (or approximations)
	function calcElbowsAndWrists (params, viewData, v)
	{
		var orig = viewData.initialPositions;
		
		if (viewData.armMeasurements === undefined) {
			viewData.armMeasurements = {
				left: measureArm(orig.Neck, orig.LeftElbow, orig.LeftWrist),
				right: measureArm(orig.Neck, orig.RightElbow, orig.RightWrist),
			};
		}
		
		calcElbowAndWrist(viewData.armMeasurements.right, params.armAngle, params.elbowBend, params.armSwing,
							v.LeftKnee, v.LeftAnkle, v.LeftHeel, v.LeftToe,	// uses opposite leg positions
							v, "RightElbow", "RightWrist");
		calcElbowAndWrist(viewData.armMeasurements.left,  params.armAngle, params.elbowBend, params.armSwing,
							v.RightKnee, v.RightAnkle, v.RightHeel, v.RightToe,
							v, "LeftElbow", "LeftWrist");
	}
	
	
	// adjust length of stride, which means scale X-position of knees & heels, and move
	//	toes & ankles the same dx as the heels -- note that this is all a hack and won't
	//	look right, especially as the strideLength adjustment moves away significantly from 1.0
	//	To do it right, it must be done in the locomotion code, where the IK comes into play.
	function strideAdjustment (params, newValues)
	{
		var mx = params.strideLength,
			anchor = newValues.Root[0];
		
		if (mx !== 1) {
			var oldValues = deepClone(newValues);
			
			Object.keys(newValues).forEach(function (key) {
				newValues[key][0] = (newValues[key][0] - anchor) * mx + anchor;
			});
			
			var dxr = newValues.RightHeel[0] - oldValues.RightHeel[0],
				dxl = newValues.LeftHeel[0] - oldValues.LeftHeel[0];
			
			//console.logToUser("old = " + JSON.stringify(oldValues));
			//console.logToUser("new = " + JSON.stringify(newValues));
			
			newValues.RightToe[0] = oldValues.RightToe[0] + dxr;
			newValues.LeftToe[0] = oldValues.LeftToe[0] + dxl;
			newValues.RightAnkle[0] = oldValues.RightAnkle[0] + dxr;
			newValues.LeftAnkle[0] = oldValues.LeftAnkle[0] + dxl;
		}
	}
	
	// when setting rotation on the handle task, it's relative to the original
	//	orienation -- for example, if the toe is drawn directly below the heel, when its
	//	position is placed to the right of the heel, in effect it has already been rotated 90
	//	degrees. So if you want to keep the toe at that angle (flat to the floor), you'd have to
	//	set it to 90. So here we measure the original angle and always set relative to that. 
	function getInitialToeAngle (bFacingLeft, heelPos, toePos)
	{
		if (heelPos && toePos) {
			var direction = bFacingLeft ? -1 : 1,
				relativeTo = [direction, 0];	// relative to right (3 o'clock) or left (9 o'clock)
			
			return (angleFromPoints(heelPos, toePos) - angleFromPoints([0, 0], relativeTo));
		} else {
			return 0;
		}
	}

	
	function setupInitialViewData (self, args, viewData)
	{
		if (viewData.initialPositions === undefined) {
			var aHandleIDs = Object.keys(viewData.handles);
			
			// only happens once ever
			var info = gatherInitialHandlePositions(args, aHandleIDs, viewData.handles);
			
			//console.logToUser(JSON.stringify(info));
			
			viewData.initialPositions	= info.positions;
			viewData.floorY				= info.floorY;
			viewData.sFloorHandleName	= info.sFloorHandleName;
			viewData.rootOffset			= info.rootOffset;
		}
	}
	
	
	function fixupHeadBang (viewData, inOutNewValues)
	{
		var initPoses = deepClone(viewData.initialPositions);
		
		// correct for slow backwards drift left/right (of all handles)
		var dx = initPoses.Root[0] - inOutNewValues.Root[0];
		
		Object.keys(inOutNewValues).forEach(function (key) {
				inOutNewValues[key][0] += dx;
		});

		// work around strange leg movement for head bang (DVACH-1037)
		kLegFieldNames.forEach(function (name) {
			var pos = initPoses[name];
			if (pos) {
				inOutNewValues[name] = pos;
			}
			//delete inOutNewValues[name]; // could do this instead, but legs might fly around
		});		
	}
	
	
	// leaves field unset if no toe or heel
	function calcToeAngle(toeAngleField, toePos, heelPos, floorY, direction, strength, toeRestAngle, inOutToeCurl)
	{
		if (toePos && heelPos) {
			var distFromFloor = floorY - toePos[1],
				kFadeDist = distanceBetween(heelPos, toePos)/5,	// guess that ball to toe length is 1/5 entire foot
				factor = 1 - ((distFromFloor < 0) ? 0 : ((distFromFloor > kFadeDist) ? 1 : (distFromFloor/kFadeDist)));
			
			// factor of 1 means on the floor, so point toe to side; factor of 0 means off the floor, point toe in
			//	direction of heel->toe (we'd prefer to just reduce the strength in this case, but that
			//	doesn't work yet)
						
			factor *= strength;

			// +/- 90 here corrects for angleFromPoints() having zero at 6 o'clock
			inOutToeCurl[toeAngleField] = Z.mathUtils.lerp((angleFromPoints(heelPos, toePos) + direction * Z.mathUtils.rad(90)),
														   0, factor) - toeRestAngle[toeAngleField];
		}
	}
	
	
	// returns toeCurl info { toe: "LeftToe" or "RightToe", angle: angle }, or empty object if not touching
	function adjustToTouchFloor(direction, toeBendStrength, viewData, inOutNewValues)
	{
		var toeCurl = {};
		
		if (viewData.floorHandleY !== undefined && viewData.floorY !== undefined) {
			var dy = viewData.floorY - viewData.floorHandleY;

			Object.keys(inOutNewValues).forEach(function (key) {
					inOutNewValues[key][1] += dy;
			});
			
			if (toeBendStrength !== 0) {
				calcToeAngle("LeftToeAngle", inOutNewValues.LeftToe, inOutNewValues.LeftHeel,
							 viewData.floorY, direction, toeBendStrength, viewData.toeRestAngle, toeCurl);
				calcToeAngle("RightToeAngle", inOutNewValues.RightToe, inOutNewValues.RightHeel,
							 viewData.floorY, direction, toeBendStrength, viewData.toeRestAngle, toeCurl);
			}
		}
		
		return toeCurl;
	}
	
	
	function facingFactor (bFacingLeft)
	{
		return bFacingLeft ? -1 : 1;
	}
	
	
	// some inefficiency here where things that could be done once per view are being
	//	done again for each view (e.g. querying params)
	// returns falsy if no handles or setup failed; otherwise returns an object with
	//	currentFrameDx field holding overall movement
	function onAnimateGuts (self, args, params, viewData, t, secondsElapsed)
	{
		var newValues = viewData.walkState.step(secondsElapsed);
				
		if (newValues.Root) {	// otherwise handle setup went badly wrong, see DVACH-1026
			
			strideAdjustment(params, newValues);
			
			calcElbowsAndWrists(params, viewData, newValues);

			if (viewData.bFacingLeft) {
				flipHorizontal(newValues);	// flip back to face left, as loco code always works to the right
			}
			
			var toeCurl = {};
			
			if (viewData.gait === kGaitStyle_HeadBang) {
				fixupHeadBang(viewData, newValues);
			} else {
                // floor adjustment depends on raw result from walkState.step() without benefit of
                //  fixupHeadBang, so it doesn't work right for HeadBang; but no problem, it also
                //  isn't needed, so we just skip it for HeadBang
                toeCurl = adjustToTouchFloor(facingFactor(viewData.bFacingLeft),
											 params.toeBend, viewData, newValues);
            }
		
			var stageLayer = args.stageLayer.privateLayer,
				handlePuppet = stageLayer.getHandleTreeRoot(),
				matScene_StageLayer = mat3.multiply(args.getLayerMatrixRelativeToScene(stageLayer),
													tasks.handle.getFrameRelativeToLayer(handlePuppet)),
				nowRootX = newValues.Root[0],
				rootDeltaX = (viewData.lastRootX === undefined) ? 0 : (nowRootX - viewData.lastRootX),
				result = {};
			
			// if this is the active view, this value will be used to adjust
			//	overallMatrixX, otherwise ignored
			result = { currentFrameDx: rootDeltaX * params.movementSpeed };

			// move values to be relative to overallMatrixX instead of root
			//	so only the _active_ view moves all views
			//	TODO: rearrange so this code & task attachment come _after_ all
			//			views are processsed; otherwise overallMatrixX is from the previous frame 
			var viewDx = -nowRootX;
			
			if (!viewData.bSetOverallMatrix) {
				viewDx += self.overallMatrixX;
			}
			Object.keys(newValues).forEach(function (key) {
				newValues[key] = vec2.add(newValues[key], viewData.rootOffset);	// back to original coord system
				newValues[key][0] += viewDx;
			});

			var aHandleIDs = Object.keys(viewData.handles);

            if (params.strength) {  // conditional shouldn't be needed here; work-around for issue where zero-strength Tasks
                                    //  aren't the same as no task at all (DVACH-1452)
                attachTasksToHandles(self, args, viewData, matScene_StageLayer,
									 aHandleIDs, newValues, toeCurl, params.strength);
            }

			viewData.lastRootX = nowRootX;
			
			return result;
		} else {
			console.logToUser("Walk behavior missing Root value");
		}
	}

	return {
		about: "Walk behavior",
		description: "$$$/animal/Behavior/Walk/Desc=Automatically walks legs drawn in profile when body parts (knee, ankle, heel, toe, etc.) are tagged",
		uiName: "$$$/animal/Behavior/Walk/UIName=Walk",
		defaultArmedForRecordOn: true,
		
		defineParams: function () {
			return [
				{
					id: "keyInput", type: "eventGraph", 
					uiName: "$$$/Behavior/Walk/Parameter/KeyboardInput=Keyboard Input",
					inputKeysArray: ["Keyboard/"],
					uiToolTip: "$$$/Behavior/Walk/Parameter/KeyboardInput/tooltip=Use left/right arrow keys to control walking",
					defaultArmedForRecordOn: true
				},
				{
					id:"start", type:"enum",
					uiName: "$$$/animal/Behavior/Walk/Parameter/Start=Start",
					items: [
						{ id: kStartImmediately,	uiName: "$$$/animal/Behavior/Walk/Parameter/Start/Immediately=Immediately" },
						{ id: kStartWithArrowKeys,	uiName: "$$$/animal/Behavior/Walk/Parameter/Start/WithArrowKeys=With Left & Right Arrow Keys" },
					],
					dephault: kStartImmediately,
					// tooltip seems unnecessary
				},
				{
					id: "ease", type: "slider",
				 	uiName: "$$$/animal/Behavior/Walk/Parameter/StartStopEasing=Start/Stop Easing",
				 	precision: 2, dephault: 0.2, uiUnits: "$$$/animal/Behavior/Dragger/units/sec=sec",
					min: 0,
					uiToolTip: "$$$/animal/Behavior/Walk/Parameter/StartStopEasing/ToolTip=How quickly speed ramps up/down when starting movement"
				},
				{
					id:"gaitStyle", type:"enum",
					uiName: "$$$/animal/Behavior/Walk/Parameter/Style=Style",
					items: aGaitStyles,
					dephault: 3,
					uiToolTip: "$$$/animal/Behavior/Walk/Parameter/Style/ToolTip=Controls the overall look of the walk cycle"
				},
				{
					id: "strideLength", type: "slider",
					uiName: "$$$/animal/Behavior/Walk/Parameter/StrideLength=Stride Length",
					precision: 0, dephault: 100, uiUnits: "%",
					min: 0,
					uiToolTip: "$$$/animal/Behavior/Walk/Parameter/StrideLength/ToolTip=Affects step spacing and arm movement; looks best between 80-120%"
				},
				{
					id:"cushion", type:"checkbox",
					uiName: "Cushion",
					dephault: false,
					hidden: true, 	// not useful enough (too subtle of a change)
					uiToolTip: "eases in & out of motion keyframes, only noticable at slow speeds; exaggerates the movement a bit"
				},
				{
					id: "speed", type: "slider",
				 	uiName: "$$$/animal/Behavior/Walk/Parameter/StepSpeed=Step Speed",
				 	precision: 0, dephault: 100, uiUnits: "%",
                    min: -9999, max: 9999,
					uiToolTip: "$$$/animal/Behavior/Walk/Parameter/StepSpeed/ToolTip=100% is two steps per second"
				},
				{
					id: "phase", type: "angle",
				 	uiName: "$$$/animal/Behavior/Walk/Parameter/StepPhase=Step Phase",
					precision: 0, dephault: 0,
					uiToolTip: "$$$/animal/Behavior/Walk/Parameter/StepPhase/ToolTip=Controls which leg starts in front; when Step Speed is 0%, animating this parameter can manually control stepping"
				},
				{
					id: "movementSpeed", type: "slider",
					uiName: "$$$/animal/Behavior/Walk/Parameter/BodySpeed=Body Speed",
					precision: 0, dephault: 0, uiUnits: "%",
					uiToolTip: "$$$/animal/Behavior/Walk/Parameter/BodySpeed/ToolTip=100% results in no foot-sliding; 0% prevents all movement"
				},
				{
					id: "armSwing", type: "slider",
					uiName: "$$$/animal/Behavior/Walk/Parameter/ArmSwing=Arm Swing",
					precision: 0, dephault: 50, uiUnits: "%",
					min: 0,
					uiToolTip: "$$$/animal/Behavior/Walk/Parameter/ArmSwing/ToolTip=Arm swing rotation as a percentage of leg swing rotation"
				},
				{
					id: "armAngle", type: "angle",
					uiName: "$$$/animal/Behavior/Walk/Parameter/ArmAngle=Arm Angle",
					precision: 0, dephault: 0,
					uiToolTip: "$$$/animal/Behavior/Walk/Parameter/ArmAngle/ToolTip=Rotation angle of arms at rest; 0^D = straight down, 90^D = straight out in front"
				},
				{
					id: "elbowBend", type: "angle",
					uiName: "$$$/animal/Behavior/Walk/Parameter/ElbowBend=Elbow Bend",
					precision: 0, dephault: 45, // 45 gets arms into reasonable default bend
					uiToolTip: "$$$/animal/Behavior/Walk/Parameter/ElbowBend/ToolTip=How much the elbows bend at rest"
				},
				{
					id: "toeBend", type: "slider", 
					uiName: "$$$/animal/Behavior/Walk/Parameter/ToeBend=Toe Bend",
					precision: 0, dephault: 50, uiUnits: "%",
					min: 0, max: 100,
					uiToolTip: "$$$/animal/Behavior/Walk/Parameter/ToeBend/ToolTip=How much the toes bend as the heel lifts"
				},
				{
					id: "strength", type: "slider",
					uiName: "$$$/animal/Behavior/Walk/Parameter/Strength=Strength",
					precision: 0, dephault: 100, uiUnits: "%",
					min: 0, max: 100,
					uiToolTip: "$$$/animal/Behavior/Walk/Parameter/Strength/ToolTip=Controls the influence of the Walk behavior on the tagged handles; 0% = rest pose, 100% = full walk cycle"
				},
				{
					// TODO: planning to remove this param (and always do kMoveEntirePuppet, once it no longer
					//	locks down children handles & fights with Transform)
					id: "overallMovement", type: "enum",
					uiName: "Overall Movement",
					items: [
						{ id: kMoveHandles,			uiName: "Move Handles" },
						{ id: kMoveEntirePuppet,	uiName: "Move Puppet" },
					],
					dephault: kMoveEntirePuppet,
					hidden: true,
					uiToolTip: "When moving the whole body, choose between moving the group that Walk is applied to, or all the handles that are tagged (will be removed when the former doesn't stick other handles in place)"
				},
				{
					id: "facing", type: "enum",
					uiName: "Drawn Facing",
					items: [
						{ id: kFacingAutomatic,		uiName: "Automatic" },
						{ uiName: "" },	// separator
						{ id: kFacingLeft,			uiName: "Left" },
						{ id: kFacingRight,			uiName: "Right" },
					],
					dephault: kFacingAutomatic,
					hidden: true, 	// hoping Automatic is so good we don't need manual
					uiToolTip: "Describe which direction the character is facing; automatic mode looks at the handles positions to guess"
				},
				{ id: "viewLayers", type: "layer", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/Views=Views",
				 	// REPEATED above; TODO: factor
				 	dephault: { match:["//Adobe.Face.LeftProfile|Adobe.Face.RightProfile", "."] }
				},
				{
					id: "handlesGroup", type: "group",
					uiName: "$$$/animal/Behavior/Walk/Parameter/HandlesGroup=Handles",
					groupChildren: defineHandleParams()
				},
			];
		},

		defineTags: function () {
			return {
				aTags: aAllTagDefinitions
			};
		},

		onCreateBackStageBehavior: function () {
		},

		onCreateStageBehavior: function (self, args) {			
			self.bKeyPreviouslyDown = false;
			self.oldDirection = 1;
			self.overallMatrixX = 0;
			
			self.aViewData = [];	// array of { handles: {}, layer, bFacingLeft }
			
			var aViews = args.getStaticParam("viewLayers");

			// first pass -- check for what each view holds, filter out ones without some matching handles
			aViews.forEach(function (view, viewIndex) {
				var bMatchedHandles = false,
					data =	{
								layer: view,	// for triggering maybe
								handles: {},
								bFacingLeft: undefined,	// filled in later (if certain), when handle positions available
								initialPositions: undefined	// ditto
					   		};
				
				aHandleTagDefinitions.forEach(function (tagDef) {
					var id = tagDef.id,
					// TODO: inefficiently calling getStaticParam inside loop
					aHandles = args.getStaticParam(makeHandleIdFromLabel(id))[viewIndex]; // returns array of handles

					if (aHandles.length > 0) {
						data.handles[id] = aHandles[0];	// we only match one per view, so this is it
						bMatchedHandles = true;			// has valid Walky handles
					}
				});
				
				if (bMatchedHandles) {
					self.aViewData.push(data);
				}
			});

			// not clear if bCheckArrows should also affect this, or if it should also trigger
			//	with a single view? If one day bCheckArrows should affect it, we'll need a clearTriggerable()
			//	SDK method.
			self.bTriggering = (self.aViewData.length > 1);	// if more than one active view, need to trigger them
			
			// If we want to only considering triggering to be enabled when there are actually
			//	_both_ left- & right-facing views, the 2nd pass must come later, during onAnimate,
			//	because that's where we gather all the initial positions and guess the view direction (which is
			//	subject to the Drawn Facing parameter). But for now, we are just considering more than one
			//	valid-Walk-handle view as being grounds for triggering.

			if (self.bTriggering) {
				self.aViewData.forEach(function (viewData) {
					var bHideSiblings = true;
					viewData.layer.setTriggerable(bHideSiblings);
				});
			}

		},

		onAnimate: function (self, args) {
			var t = args.t + args.globalRehearseTime,
				bSingleView = !self.bTriggering,
				bViewHasBeenTriggered = false;
						
			if (self.old_t === undefined) {
				self.old_t = 0;
				self.oldPhase = 0;
			}
			
			// phaseChange of 1 sec works out to one cycle exactly because at 100% step speed
			//	your get one cycle/second (2 steps/sec)
			var phase = args.getParam("phase")/360,
				phaseChange = phase - self.oldPhase,
				userDirection = getUserCommandedDirection(self, args, t),
				dt = (t - self.old_t),
				params = {		// shared by all views
					movementSpeed:	args.getParam("movementSpeed")/100,
					strength:		args.getParam("strength")/100 * userDirection.strength,
					armAngle:		-args.getParam("armAngle"),		// positive value brings arms up in front, 0 = down
					elbowBend:		args.getParam("elbowBend"),		// positive value bends more, 0 = no bend
					armSwing: 		args.getParam("armSwing")/100,
					strideLength:	args.getParam("strideLength")/100,
					toeBend:		args.getParam("toeBend")/100
				};
			
			if (params.toeBend > 0) {
				self.bHasEverBentToes = true;	// workaround to deal with never being able to not set rotation
												//	once it's ever been set
			} else if (self.bHasEverBentToes) {
				params.toeBend = 0.01;	// any non-zero small value will work here
			}
						
			self.aViewData.forEach(function (viewData) {
				setupInitialViewData(self, args, viewData);

				var phaseAdjustment = setupGait(self, args, viewData),	// sets viewData.bFacingLeft & other things
					direction = userDirection.direction;
					
				// to make arrow direction match character direction
				if (userDirection.bCheckArrows && viewData.bFacingLeft) {
					direction *= -1;
					phaseChange *= -1;	// fixes DVACH-1343 -- keeping same leg stationary when switching directions
					// now direction can be thought of as in the loco-space, i.e. positive is walking
					//	forward
				}
				
				var p = viewData.initialPositions;

				viewData.toeRestAngle = {
					LeftToeAngle:	getInitialToeAngle(viewData.bFacingLeft, p.LeftHeel, p.LeftToe),
					RightToeAngle:	getInitialToeAngle(viewData.bFacingLeft, p.RightHeel, p.RightToe),
				};
				
				var secondsElapsed = dt * userDirection.stepSpeed * direction + phaseChange + phaseAdjustment;
				
				// don't have a way to identify _which_ view tag was applied (no tag access
				//	to behaviors yet) -- would be handy to know left vs. right profile
				//	view, but our heuristics are OK for now
				
				var viewResult = onAnimateGuts(self, args, params, viewData, t, secondsElapsed);
				
				if (viewResult) {					
					var bThisViewActive = bSingleView;

					if (!bThisViewActive) {
						bThisViewActive = (userDirection.direction === facingFactor(viewData.bFacingLeft));
					}

					if (bThisViewActive && !bViewHasBeenTriggered) {
						bViewHasBeenTriggered = true;
						// apply global move & trigger (only once, even if multiple report as being active)
						self.overallMatrixX += viewResult.currentFrameDx * params.strideLength;

						if (self.bTriggering) {
							var priority = 0.7;	// not clear what would be best here
							viewData.layer.trigger(priority);
						}
					}
				}
			});

			var bSetOverallMatrix = (args.getParam("overallMovement") === kMoveEntirePuppet);
			if (bSetOverallMatrix) {
				//console.logToUser("setFrame(x) = " + self.overallMatrixX);
				tasks.handle.moveFrameBy(args.stageLayer.privateLayer.getHandleTreeRoot(),
						  Z.Mat3.identity().translate([self.overallMatrixX, 0]));
			}

			self.old_t = t;
			self.oldPhase = phase;
		}
	}; // end of object being returned
});
